/*
 * Copyright 2010 Freescale Semiconductor, Inc.
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */

#include <common.h>
#include <asm/io.h>

#include "ics307_clk.h"

#ifdef CONFIG_FSL_NGPIXIS
#include "ngpixis.h"
#else
#include "pixis.h"
#endif

/* decode S[0-2] to Output Divider (OD) */
static u8 ics307_s_to_od[] = {
	10, 2, 8, 4, 5, 7, 3, 6
};

/*
 * Calculate frequency being generated by ICS307-02 clock chip based upon
 * the control bytes being programmed into it.
 */
static unsigned long ics307_clk_freq(u8 cw0, u8 cw1, u8 cw2)
{
	const unsigned long input_freq = CONFIG_ICS307_REFCLK_HZ;
	unsigned long vdw = ((cw1 << 1) & 0x1FE) + ((cw2 >> 7) & 1);
	unsigned long rdw = cw2 & 0x7F;
	unsigned long od = ics307_s_to_od[cw0 & 0x7];
	unsigned long freq;

	/*
	 * CLK1 Freq = Input Frequency * 2 * (VDW + 8) / ((RDW + 2) * OD)
	 *
	 * cw0:  C1 C0 TTL F1 F0 S2 S1 S0
	 * cw1:  V8 V7 V6 V5 V4 V3 V2 V1
	 * cw2:  V0 R6 R5 R4 R3 R2 R1 R0
	 *
	 * R6:R0 = Reference Divider Word (RDW)
	 * V8:V0 = VCO Divider Word (VDW)
	 * S2:S0 = Output Divider Select (OD)
	 * F1:F0 = Function of CLK2 Output
	 * TTL = duty cycle
	 * C1:C0 = internal load capacitance for cyrstal
	 *
	 */

	freq = input_freq * 2 * (vdw + 8) / ((rdw + 2) * od);

	debug("ICS307: CW[0-2]: %02X %02X %02X => %lu Hz\n", cw0, cw1, cw2,
			freq);
	return freq;
}

unsigned long get_board_sys_clk(void)
{
	return ics307_clk_freq(
			in_8(&pixis->sclk[0]),
			in_8(&pixis->sclk[1]),
			in_8(&pixis->sclk[2]));
}

unsigned long get_board_ddr_clk(void)
{
	return ics307_clk_freq(
			in_8(&pixis->dclk[0]),
			in_8(&pixis->dclk[1]),
			in_8(&pixis->dclk[2]));
}
