#include <err.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <util.h>
#include <time.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "liblif.h"


#define DEFAULT_LAST_VOL_FLAG	1
#define DEFAULT_VOL_NUM		1

#define FILETYPE_BINARY		1
#define FILETYPE_ASCII		0000
#define FILETYPE_BIN		-23951

static void usage(void);
static int check_lifvol(char *volname);
static int check_regfile(char *filename);
static void copyfile(char *srcvol, char *srcfile, char *targvol, char *targfile);
static int lif_insertdir(int fd, struct lifdir *ld);


extern char *__progname;

static int aflag, bflag, rflag, tflag, Tflag;
static int filetype;
static int lastvolflag;
static int volnum;
static int implementation;
static int target;
#define TARG_DIR		1
#define TARG_FILE		2
#define TARG_LIFVOL		3
#define FULLNAMELEN		2048
static char fullname[FULLNAMELEN];

int
main(int argc, char *argv[])
{
	int ch;
	struct stat fs;
	char *targ_volname, *targ_filename, *src_volname, *src_filename;
	char *ptr;
	int i;

	aflag = bflag = rflag = tflag = Tflag = 0;
	filetype = FILETYPE_BINARY;
	lastvolflag = DEFAULT_LAST_VOL_FLAG;
	volnum = DEFAULT_VOL_NUM;
	while ((ch = getopt(argc, argv, "abi:rtv:L:T:")) != -1) {
		switch (ch) {
		case 'a':			/* ASCII mode */
			aflag = 1;
			break;
		case 'b':			/* binary mode */
			bflag = 1;
			break;
		case 'i':			/* "implementation" field */
			implementation = atoi(optarg);
			break;
		case 'r':			/* raw mode */
			rflag = 1;
			break;
		case 't':			/* convert filenames */
			tflag = 1;
			break;
		case 'v':			/* volume number */
			volnum = atoi(optarg);
			break;
		case 'L':			/* last volume flag */
			lastvolflag = (atoi(optarg) != 0);
			break;
		case 'T':			/* file type */
			Tflag = 1;
			filetype = atoi(optarg);
			break;
		default:
			usage();
			/* NOTREACHED */
		}
	}
	argc -= optind;
	argv += optind;

	if (argc < 2)
		usage();

	if (aflag + bflag + rflag + Tflag > 1)
		usage();

	/*
	 *  Check if the target is a Unix file, Unix directory
	 *  or LIF file.
	 */
	lif_parse_filename(argv[argc-1], &targ_volname, &targ_filename);
	if (targ_volname != NULL) {
		/*
		 * Check lif volume is valid.
		 */
		if (check_lifvol(targ_volname))
			err(1, "not valid LIF volume %s", targ_volname);
		target = TARG_LIFVOL;
	} else {

		/*
		 * Not a lif volume, check if it's a Unix directory.
		 */
		if (stat(targ_filename, &fs) < 0)
			target = TARG_FILE;
		else
			target = (fs.st_mode & S_IFMT) == S_IFDIR ?
			    TARG_DIR : TARG_FILE;
	}

	if (target == TARG_FILE) {
		printf("1\n");
		/*
		 * Source must be a single lif file.
		 */
		if (argc != 2)
			usage();
		lif_parse_filename(argv[0], &src_volname, &src_filename);
		if (src_volname == NULL)
			errx(1, "%s not a lif volume", argv[0]);
		if (check_lifvol(src_volname))
			errx(1, "%s not a valid lif file", src_volname);
		copyfile(src_volname, src_filename, NULL, targ_filename);
		return (0);
	}

	if (target == TARG_DIR) {
		printf("2\n");
		/*
		 * Sources must be lif files.
		 */
		for (i=0; i<argc-1; i++) {
			lif_parse_filename(argv[i], &src_volname,
			    &src_filename);
			if (src_volname == NULL)
				err(1, "%s not a LIF volume", argv[i]);
			strncpy(fullname, targ_filename, FULLNAMELEN);
			strncat(fullname, "/", FULLNAMELEN);
			strncat(fullname, src_filename, FULLNAMELEN);
			copyfile(src_volname, src_filename, NULL, fullname);
		}
	}

	if ((target == TARG_LIFVOL) && (targ_filename != NULL)) {
		printf("3\n");
		/*
		 * Can only be a single source file
		 */
		if (argc != 2)
			usage();
		lif_parse_filename(argv[0], &src_volname, &src_filename);
		copyfile(src_volname, src_filename, NULL, targ_filename);
		return (0);
	}

	if (target == TARG_LIFVOL) {
		printf("4\n");

		/*
		 * Sources can be lif files or Unix files.
		 */

		for (i=0; i<argc-1; i++) {
			lif_parse_filename(argv[i], &src_volname,
			    &src_filename);
			if (src_volname == NULL) {
				if (check_regfile(src_filename))
					continue;
				ptr = strrchr(src_filename,'/');
				ptr = (ptr == NULL ? src_filename : ptr+1);
				copyfile(src_volname, src_filename,
				    targ_volname, ptr);
			} else {
				if (check_lifvol(src_volname))
					continue;
				copyfile(src_volname, src_filename,
				    targ_volname, src_filename);
			}
		}
	}

	return (0);
}

static void
usage(void)
{

	fprintf(stderr, "Usage: %s [-a | -b | -r] [-t] [-i impl] [-L flag] [-T type] [-v volnum] file1 file2\n", __progname);
	fprintf(stderr, "Usage: %s [-a | -b | -r] [-t] [-i impl] [-L flag] [-T type] [-v volnum] file1 ... directory\n", __progname);

	exit(EXIT_FAILURE);
}

static int
check_lifvol(char *volname)
{
	int fd;
	struct lifvol lv;
	int error = 0;

	if ((fd = lif_openvol(volname, O_RDONLY)) < 0)
		return (-1);
	error = (read(fd, &lv, sizeof(struct lifvol))
	    != sizeof(struct lifvol));
	close(fd);
	return ((error || (be16toh(lv.lv_magic) != VOL_MAGIC)) ? -1 : 0);
}

static int
check_regfile(char *filename)
{
	struct stat fs;

	if (filename == NULL)
		return (-1);

	if (strncmp(filename, "-", 1) == 0)
		return (0);

	if (stat(filename, &fs) < 0)
		return (-1);
	if ((fs.st_mode & S_IFMT) != S_IFREG)
		return (-1);
	if (check_lifvol(filename) == 0)
		return (-1);
	return (0);
}



static void
copyfile(char *srcvol, char *srcfile, char *targvol, char *targfile)
{
	char liftargfile[FULLNAMELEN];
	u_int8_t buf[SECTOR_SIZE];
	int n, i, flags;
	int src_is_targ = 0;
	struct lifdir src_ld, targ_ld;
	struct stat fs;
	struct tm tm, *tmp;
	int src_fd, targ_fd;
	int size, type, flag;

	printf("copyfile: %s:%s -> %s:%s\n",
	    srcvol, srcfile, targvol, targfile);
	
	if (targvol != NULL) {
		/*
		 * Sanitise filenames destined for LIF volume
		 */
		if (tflag)
			lif_makename(targfile, liftargfile);
		else {
			strncpy(liftargfile, targfile, 6);
			n = strlen(liftargfile);
			for (; n<10; n++)
				liftargfile[n] = '\0';
			if (lif_checkname(liftargfile))
				errx(1, "invalid filename %s", targfile);
		}
	} else
		strncpy(liftargfile, targfile, FULLNAMELEN);

	if (srcvol != NULL && lif_checkname(srcfile))
		errx(1, "invalid filename %s", srcfile);

	printf("copying from %s:%s to %s:%s\n",
		srcvol, srcfile, targvol, liftargfile);

	/*
	 * Check if source and target volume are the same.
	 */
	if (srcvol != NULL && targvol != NULL) {
		if (strcmp(srcvol, targvol) == 0)
			src_is_targ = 1;
	}

	if (srcvol != NULL) {
		flags = (src_is_targ ? O_RDONLY : O_RDWR);
		src_fd = lif_openvol(srcvol, flags);
		if (lif_lookup(src_fd, srcfile, &src_ld) < 0)
			errx(1, "cannot find file %s in volume %s",
			    srcfile, srcvol);
		if (lseek(src_fd, be32toh(src_ld.ld_addr), SEEK_SET) < 0)
			err(1, "cannot seek");
		size = be32toh(src_ld.ld_length);
		lif_gettime(&src_ld.ld_toc, &tm);
		tmp = &tm;
		type = be16toh(src_ld.ld_type);
		flag = DIR_FLAG;	/* XXX */
	} else {
		src_fd = open(srcfile, O_RDONLY);
		if (fstat(src_fd, &fs) < 0)
			err(1, "cannot stat");
		size = fs.st_size;
		tmp = gmtime(&fs.st_ctime);
		type = DIR_TYPE;
		flag = DIR_FLAG;
	}

	/*
	 * We're now pointing at the start of the data to transfer
	 * in the source file.
	 */

	if (targvol != NULL) {
		if (src_is_targ)
			targ_fd = src_fd;
		else
			targ_fd = lif_openvol(targvol, O_RDWR);

		if (lif_lookup(targ_fd, liftargfile, &targ_ld) >= 0)
			errx(1, "%s would overwrite existing file %s",
			    targfile, liftargfile);
		/*
		 * Insert a new directory entry
		 */
		strncpy(targ_ld.ld_name, liftargfile, 10);
		targ_ld.ld_type = htobe16(type);
		targ_ld.ld_flag = htobe16(flag);
		targ_ld.ld_length = htobe32(size);
		lif_puttime(&targ_ld.ld_toc, tmp);
		if (lif_insertdir(targ_fd, &targ_ld))
			errx(1, "cannot write directory entry");
		printf("seeking to %d\n", be32toh(targ_ld.ld_addr));
		if (lseek(targ_fd, be32toh(targ_ld.ld_addr), SEEK_SET) < 0)
			err(1, "cannot seek");
	} else {
		if ((targ_fd = open(targfile, O_CREAT|O_WRONLY|O_TRUNC,
		    0644)) < 0)
			errx(1, "can't create file %s", targfile);
	}

	/*
	 * Do the copy.
	 */
	printf("copying %d bytes\n", size);
	do {
		n = (size < SECTOR_SIZE ? size : SECTOR_SIZE);
		i = read(src_fd, buf, n);
		if (i != n)
			errx(1, "read failed");
		i = write(targ_fd, buf, n);
		if (i != n)
			errx(1, "write failed");
		size -= n;
	} while (size > 0);

	close(src_fd);
	close(targ_fd);
}

static int
lif_insertdir(int fd, struct lifdir *rld)
{
	struct lifdir *ld;
	struct lifvol lv;
	u_int8_t buf[SECTOR_SIZE];
	size_t addr, offset;
	int blocknum, i;
	int freedir, found;

	printf("lif_insertdir: fd=%d, ld=%p, name=%s\n",
	    fd, rld, rld->ld_name);

	/* XXX check that the space exists for the file */

	/* the address of the space to write into is
	 * filled into the original lifdir structure
	 */

	lseek(fd, 0, SEEK_SET);
	if (read(fd, &lv, sizeof(struct lifvol)) != sizeof(struct lifvol))
		return (-1);

	blocknum = be32toh(lv.lv_dirstart);
	addr = blocknum * SECTOR_SIZE +
	    sizeof(struct lifvol)*be32toh(lv.lv_dirsize);
	freedir = found = 0;
	for (i=0; i<htobe32(lv.lv_dirsize); i++) {

		if ((i % (SECTOR_SIZE/sizeof(struct lifdir))) == 0) {
			lif_block_read(fd, blocknum++, buf);
			ld = (struct lifdir *)buf;
		}
		if (be16toh(ld->ld_type) == 0 && !found) {
			found = 1;
			freedir = i;
			continue;
		}
		if (be16toh(ld->ld_type) == 0xffff && !found) {
			found = 1;
			freedir = i;
			break;
		}

		if (be16toh(ld->ld_addr) + be16toh(ld->ld_length) > addr)
			addr =	be16toh(ld->ld_addr) + be16toh(ld->ld_length);
		ld++;
	}

	printf("freedir=%d, addr=%d found=%d\n", freedir, addr, found);

	if (!found)
		return (-1);

	printf("volsize %d, addr %d, size %d\n", be32toh(lv.lv_length),
		addr, be32toh(rld->ld_length));

	/* check if we have enough room */
	if (htobe32(lv.lv_length) < addr + be32toh(rld->ld_length)) {
		printf("not enough room");
		return (-1);
	}

	offset = be32toh(lv.lv_dirstart)*SECTOR_SIZE +
	    freedir*sizeof(struct lifdir);

	printf("offset = %d\n", offset);

	if (be16toh(ld->ld_type) == 0xffff) {

		/*
		 * Keep the end-of-directory marker, or move it to the
		 * next entry
		 */

		if (be32toh(lv.lv_dirsize) != freedir) {
			lseek(fd, offset + sizeof(struct lifdir), SEEK_SET);
			memset(buf, 0, sizeof(struct lifdir));
			ld = (struct lifdir *)buf;
			ld->ld_type = htobe16(0xffff);
			if (write(fd, ld, sizeof(struct lifdir)) !=
			    sizeof(struct lifdir))
				return (-1);
		} else
			rld->ld_type = htobe16(0xffff);
	}

	rld->ld_addr = htobe32(addr);

	lseek(fd, offset, SEEK_SET);
	if (write(fd, rld, sizeof(struct lifdir)) != sizeof(struct lifdir))
		return (-1);

	return (0);
}
