/*
 * Copyright (C) 2006-2008 the VideoLAN team
 *
 * This file is part of VLMa.
 *
 * VLMa is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * VLMa is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with VLMa. If not, see <http://www.gnu.org/licenses/>.
 *
 */

package org.videolan.vlma.order;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintWriter;
import java.net.Socket;

import org.apache.commons.configuration.Configuration;
import org.apache.log4j.Logger;
import org.videolan.vlma.model.Adapter;
import org.videolan.vlma.model.Order;
import org.videolan.vlma.model.Program;
import org.videolan.vlma.model.Server;
import org.videolan.vlma.model.StreamingStrategy;

/**
 * This class represents an order given to a VLC server.
 * It is associated with a media group.
 *
 * @author Sylvain Cadilhac <sylv at videolan.org>
 */
public abstract class OrderSender {

    private static final Logger logger = Logger.getLogger(OrderSender.class);

    /**
     * Time to wait for during telnet connections.
     */
    private static final int TELNET_WAIT_INTERVAL = 100;
    private static final int TELNET_WAIT_MAX = 5000;

    protected CommandLogger commandLogger;

    protected Configuration configuration;

    private Socket telnetSocket;

    private PrintWriter telnetOut;

    private BufferedReader telnetIn;

    /**
     * Gets the VLM command name associated with an order.
     *
     * @return the command name
     */
    protected String getVLMCommandName(Order order) {
        return "flux-" + order.getAdapter().getName();
    }

    /**
     * Connects to the VLC server using telnet.
     *
     * @throws IOException Connection aborted.
     */
    protected void telnetConnect(Server server) throws IOException {
        telnetSocket = new Socket(server.getIp(), configuration.getInt("vlc.telnet.port"));
        telnetOut = new PrintWriter(telnetSocket.getOutputStream(), true);
        telnetIn = new BufferedReader(new InputStreamReader(telnetSocket.getInputStream()));
        telnetOut.println(configuration.getString("vlc.telnet.password"));
        // Wait for the telnet session to be initialized before sending commands
        try {
            Thread.sleep(configuration.getInt("vlc.telnet.delay"));
        } catch (InterruptedException e) {
        }
    }

    /**
     * Closes the telnet connection.
     *
     * @throws IOException Operation aborted.
     */
    protected void telnetClose() throws IOException {
        telnetOut.close();
        telnetIn.close();
        telnetSocket.close();
    }

    /**
     * Sends a command to to VLC server.
     *
     * @param command the command to send
     * @throws IOException Operation aborted.
     */
    protected void telnetCommand(String command, Adapter adapter) throws IOException {
        synchronized (adapter.getServer()) {
            logger.debug("Send command " + command + " to " + adapter.getServer().getName());
            telnetOut.println(command);
            try {
                Thread.sleep(configuration.getInt("vlc.telnet.delay"));
            } catch (InterruptedException e) {
            }
            int timeSleeped = 0;
            // Continue to wait if the telnet interface is not ready yet, but
            // not more than TELNET_WAIT_MAX ms.
            while (!telnetIn.ready() && (timeSleeped += TELNET_WAIT_INTERVAL) <= TELNET_WAIT_MAX) {
                try {
                    Thread.sleep(TELNET_WAIT_INTERVAL);
                } catch (InterruptedException e) {
                }
            }
            StringBuilder response = new StringBuilder();
            while (telnetIn.ready()) {
                response.append((char) telnetIn.read());
            }
            commandLogger.add(adapter.getServer(), command, response.toString());
            logger.debug("Command result: " + response);
        }
    }

    /**
     * Stops streaming and sets up common parameters.
     *
     * @throws IOException Operation aborted
     */
    protected void init(String commandName, Adapter adapter) throws IOException {
        // Remove the stream if there is one
        telnetCommand("setup " + commandName + " disabled", adapter);
        telnetCommand("control " + commandName + " stop", adapter);
        telnetCommand("del " + commandName, adapter);
        // Create a multicast diffusion program
        telnetCommand("new " + commandName + " broadcast", adapter);
        // Verbose mode using colors
        telnetCommand("setup " + commandName + " option vvv", adapter);
        telnetCommand("setup " + commandName + " option color", adapter);
        telnetCommand("setup " + commandName + " option ttl="
                + Integer.toString(configuration.getInt("vlc.stream.ttl")), adapter);
    }

    /**
     * Sends the order.
     *
     * @throws IOException Operation aborted.
     */
    public abstract void start(Order order) throws IOException;

    /**
     * Gets the access method of the stream.
     *
     * @param program the program to stream
     * @return the access method
     */
    public String getAccess(Program program) {
        StreamingStrategy streamingStrategy = program.getStreamingStrategy();
        if (streamingStrategy.getProtocol().equals(StreamingStrategy.Protocol.UDP_MULTICAST)) {
            return "udp";
        } else if (streamingStrategy.getProtocol().equals(StreamingStrategy.Protocol.HTTP)) {
            return "http";
        }
        return null;
    }

    /**
     * Gets the destination of the output.
     *
     * @param program the program to stream
     * @return the destionation of the output
     */
    public String getDst(Program program) {
        StreamingStrategy streamingStrategy = program.getStreamingStrategy();
        if (streamingStrategy.getProtocol().equals(StreamingStrategy.Protocol.UDP_MULTICAST)) {
            return program.getIp().getHostAddress();
        } else if (streamingStrategy.getProtocol().equals(StreamingStrategy.Protocol.HTTP)) {
            Integer port = configuration.getInt("vlma.streaming.http.port");
            return program.getPlayer().getHostAddress() + ":" + port + "/" + program.hashCode();
        }
        return null;
    }

    /**
     * Gets the program encapsulation.
     *
     * @param program the program to stream
     * @return the encapsulation
     */
    public String getMux(Program program) {
        return program.getStreamingStrategy().getEncapsulation().toString().toLowerCase();
    }

    protected void stopCommand(String commandName, Adapter adapter) throws IOException {
        telnetCommand("control " + commandName + " disabled", adapter);
        telnetCommand("control " + commandName + " stop", adapter);
        telnetCommand("del " + commandName, adapter);
    }

    /**
     * Orders a server to stop streaming.
     *
     * @throws IOException
     */
    public void stop(Order o) throws IOException {
        telnetConnect(o.getAdapter().getServer());
        stopCommand(getVLMCommandName(o), o.getAdapter());
        telnetClose();
    }

    /**
     * Sets the configuration.
     *
     * @param configuration the configuration to set
     */
    public void setConfiguration(Configuration configuration) {
        this.configuration = configuration;
    }

    /**
     * Sets the command logger.
     *
     * @param commandLogger the commandLogger to set
     */
    public void setCommandLogger(CommandLogger commandLogger) {
        this.commandLogger = commandLogger;
    }

}
